import LicenseSpring

import SwiftUI

struct LicenseAdditionalInfoView: View {
    let license: License?
    
    @EnvironmentObject private var viewUpdater: ViewUpdaterHack
        
    var body: some View {
        VStack(spacing: 20) {
            GroupBox(label: Text("Installation Files")) {
                InstallationFilesView(license: license, appVersion: license?.configuration.appVersion ?? "-")
                    .frame(maxWidth: .infinity)
            }
            GroupBox(label: Text("License Features")) {
                LicenseFeaturesView(license: license)
                    .frame(maxWidth: .infinity)
            }
            GroupBox(label: Text("Custom Data Fields")) {
                CustomDataFieldsView(license: license)
                    .frame(maxWidth: .infinity)
            }
        }
    }
}

private struct InstallationFilesView: View {
    let license: License!
    let appVersion: String
    
    @State private var files: [RandomIdentifiable<InstallationFile>] = []
    @State private var selectionIdx: Int?
    private var selection: UUID? {
        guard let selectionIdx = selectionIdx else { return nil }
        return selectionIdx < files.count ? files[selectionIdx].id : nil
    }
    
    @EnvironmentObject private var viewUpdater: ViewUpdaterHack
    
    var body: some View {
        VStack {
            InstallationFilesTable(values: files, selection: $selectionIdx)
            HStack {
                MinWidthButton("Load App Versions") {
                    performAction(name: "Load App Versions", successAlert: false) {
                        let versions = try license.requestAvailableVersions()
                        files = versions
                            .map(\.version)
                            .sorted { $0.compare($1, options: .numeric) == .orderedDescending }
                            .map {
                                do {
                                    return try license.requestInstallationFile(version: $0)
                                } catch {
                                    print("Failed to request installation file for version \($0)")
                                    return InstallationFile(hashMD5: "-", version: $0, installationFileLink: "-")
                                }
                            }
                            .map(RandomIdentifiable.init(value:))
                    }
                }
                Spacer()
                MinWidthButton("Check For Updates") {
                    guard let version = files.first(where: { $0.id == selection })?.value else { return }
                    let text: String
                    if appVersion.compare(version.version) == .orderedAscending {
                        text = "There is a new version of the application available"
                    } else {
                        text = "Your app is up to date"
                    }
                    showAlert(title: "Check For Updates", text: text, style: .informational)
                }
                .disabled(selection == nil)
            }
            Text("Current application version: \(appVersion)")
                .frame(maxWidth: .infinity, alignment: .leading)
        }
    }
}

private struct LicenseFeaturesView: View {
    let license: License!
    
    @State private var features: [RandomIdentifiable<LicenseFeature>] = []
    @State private var selectionIdx: Int?
    private var selection: UUID? {
        guard let selectionIdx = selectionIdx else { return nil }
        return selectionIdx < features.count ? features[selectionIdx].id : nil
    }
    @State private var consumptionDelta = 0
    
    @EnvironmentObject private var viewUpdater: ViewUpdaterHack
    
    var body: some View {
        VStack {
            LicenseFeaturesTable(values: features, selection: $selectionIdx)
            HStack {
                MinWidthButton("Sync Consumption") {
                    performAction(name: "Sync Consumption") {
                        let feature = features.first { $0.id == selection }?.value
                        try license.syncFeatureConsumption(code: feature?.code)
                        updateFeatures()
                    }
                }
                Spacer()
                MinWidthButton("Update Consumption") {
                    performAction(name: "Update Consumption") {
                        guard let feature = features.first(where: { $0.id == selection })?.value else { return }
                        try license.updateFeatureConsumption(code: feature.code, difference: consumptionDelta)
                        consumptionDelta = 0
                        updateFeatures()
                    }
                }
                .disabled(selection == nil || consumptionDelta == 0)
                Stepper("\(consumptionDelta)", value: $consumptionDelta)
            }
        }
        .onAppear { updateFeatures() }
    }
    
    private func updateFeatures() {
        features = license?.features.map(RandomIdentifiable.init(value:)) ?? []
    }
}

private struct CustomDataFieldsView: View {
    let license: License?
    
    var body: some View {
        CustomFieldsTable(values: fields, selection: .constant(nil))
    }
    
    private var fields: [RandomIdentifiable<CustomField>] {
        license?.customFields.map(RandomIdentifiable.init(value:)) ?? []
    }
}

struct LicenseAdditionalInfoView_Previews: PreviewProvider {
    static var previews: some View {
        LicenseAdditionalInfoView(license: nil)
    }
}
